/* 
 * msgesrv.c
 *
 * Моделює роботу сервера, який отримує клієнтські запити через чергу повідомлень 
 * System V. Для обслуговування кожного запиту створюється дочірній процес. Обслуговування
 * зводиться до виведення запиту в стандартний потік виведення результатів. В програмі
 * реалізовано асинхронне видалення дескрипторів дочірніх процесів шляхом перехоплення
 * сигналу SIGCHLD. Перехоплюється також сигнал SIGTERM, по отриманню якого програма
 * видаляє чергу повідомлень і завершує роботу.
 * Ілюструє порядок застосування черг повідомлень System V.
 *
 */

#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include "msge.h"

#define ERRLOG "msgeserv.err"   /* Журнал повідомлень про помилки */

static int msgd = 0;            /* Дескриптор черги повідомлень */


/* Видаляє чергу повідомлень (якщо вона існує) і завершує роботу.
   exit_code - код завершення програми */
void done(int exit_code)
{
        if ((msgd > 0) && 
                (msgctl(msgd, IPC_RMID, 0) != 0) &&
                                        (exit_code == 0)) {
                perror("msgctl()");
                exit_code = EXIT_FAILURE;
        }
        exit(exit_code);
}

/* Обробник сигналу SIGTERM */
void sigterm_handler()
{
        done(EXIT_SUCCESS);
}

/* Обробник сигналу SIGCHLD */
void sigchld_handler()
{
        wait(0);
}

int main()
{
        struct msgbuf buf;      /* Буфер для повідомлення */
        /* Покажчик на поле даних повідомлення */
        struct mtext *mtext = (struct mtext *) buf.mtext;
        struct sigaction act;
        key_t msgkey;           /* Ключ черги повідомлень */

        /* Зв'язує стандартний потік виведення повідомлень про помилки
           з файлом ERRLOG (потрібно, оскільки може працювати в фоновому
           режимі). */
        if (freopen(ERRLOG, "w", stderr) == NULL) {
                perror("freopen(ERRLOG, ...)");
                exit(EXIT_FAILURE);
        }
        /* Реєструє обробники сигналів SIGTERM, SIGCHLD. */
        memset(&act, 0, sizeof(act));
        act.sa_handler = sigterm_handler;	
        if (sigaction(SIGTERM, &act, NULL) != 0) {
                perror("sigaction(SIGTERM, ...)");
                exit(EXIT_FAILURE);
        }
        act.sa_handler = sigchld_handler;	
        if (sigaction(SIGCHLD, &act, NULL) != 0) {
                perror("sigaction(SIGCHLD, ...)");
                exit(EXIT_FAILURE);
        }
        /* Генерує ключ черги повідомлень. */
        msgkey = ftok(PROJNAME, PROJNUM);
        if (msgkey == -1) {
                perror("ftok()");
                exit(EXIT_FAILURE);
        }
        /* Створює/відкриває чергу повідомлень. */
        msgd = msgget(msgkey, PERM|IPC_CREAT);
        if (msgd == -1) {
                perror("msgget()");
                exit(EXIT_FAILURE);
        }

        while (1) {   
                int rval;

                /* Отримує з черги повідомлень черговий клієнтський запит
                   (цикл потрібен для того, щоб перезапустити функцію
                   msgrcv(), коли її виконання переривається сигналом
                   (SIGCHLD)). */
                do { 
                        rval = msgrcv(msgd, &buf, FULL_TEXT_SIZE,
                                                        SERVER_ID, 0);
                } while ((rval < 0) && (errno == EINTR));
                if (rval < 0) {
                        perror("msgrcv()");
                        done(EXIT_FAILURE);
                }
                /* Створює дочірній сервер (для обслуговування щойно
                   отриманого запиту). */
                switch (fork()) {
                case -1:
                        perror("fork()");
                        done(EXIT_FAILURE);
                case 0:
                        /* Дочірній сервер */	
                        /* Просто виводить запит у стандартний потік
                           виведення результатів і завершує роботу. */
                        printf("Received: %s\n", mtext->text);
                        exit(EXIT_SUCCESS);
                }
        }
}
